/*
  Network Simulation Cradle
  Copyright (C) 2003-2005 Sam Jansen

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the Free
  Software Foundation; either version 2 of the License, or (at your option)
  any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc., 59
  Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

/* $Id: support2.c 839 2005-05-18 08:11:31Z stj2 $ */
#include <sys/param.h>
#include <sys/socket.h>
#include <sys/socketvar.h>
#include <sys/sockio.h>
#include <sys/callout.h>
#include <sys/mbuf.h>
#include <net/if.h>
#include <net/if_types.h>
#include <netinet/in.h>
#include <netinet/in_pcb.h>
#include <netinet/tcp_var.h>
#include <sys/malloc.h>
#include <sys/libkern.h>

#include <arpa/inet.h>

#include "num_stacks.h"
#include "support/support.h"

/* exported from support/interface.c */
void
fake_ether_attach(struct ifnet *ifp, unsigned int addr, unsigned int mask, int);

extern int  inet_aton(const char *cp, struct in_addr *pin);
extern void nsc_assert(int);


void *new_interface(unsigned int addr, unsigned int mask, int mtu)
{
    struct ifnet *ifp = (struct ifnet *)malloc(sizeof(struct ifnet), 0, 0);

    memset(ifp, 0, sizeof(struct ifnet));

    // Tell the stack we have a new interface
    fake_ether_attach(ifp, addr, mask, mtu);

    return ifp;
}

void tcplisten(void *socket, int port)
{
    struct socket *so = (struct socket *)socket;
    struct sockaddr_in nam;
    int error;

    // Bind socket
    memset(&nam, 0, sizeof(struct sockaddr_in));
    nam.sin_len = sizeof(struct sockaddr_in);
    nam.sin_family = AF_INET;
    nam.sin_port = htons((short)port);
    nam.sin_addr.s_addr = INADDR_ANY;

    error =
        sobind(so, (struct sockaddr *)&nam, curthread);

    assert(!error);

    // set the socket to be in a listening state
    error =
        solisten(so, 5, curthread);

    assert(!error);

    // The socket still needs to be set into an accepting state at this
    // point.
}

int tcp_accept_low(struct socket *so, struct socket **s, struct sockaddr **nam)
{
    int error = 0;
	struct socket *ts;

	if(!so)
		return ENOTCONN;
	
	if (TAILQ_EMPTY(&so->so_comp) && so->so_error == 0)
		return EAGAIN;

	if(so->so_error) {
		error = so->so_error;
		so->so_error = 0;
		return error;
	}

	/*
	 * At this point we know that there is at least one connection
	 * ready to be accepted. Remove it from the queue prior to
	 * allocating the file descriptor for it since falloc() may
	 * block allowing another process to accept the connection
	 * instead.
	 */
	ts = TAILQ_FIRST(&so->so_comp);
	TAILQ_REMOVE(&so->so_comp, ts, so_list);
	so->so_qlen--;

	/* connection has been removed from the listen queue */
	KNOTE_UNLOCKED(&so->so_rcv.sb_sel.si_note, 0);

	/* ts->so_state &= ~SS_COMP; */ /* Remove for 5.3 */
	ts->so_head = NULL;

	error = soaccept(ts, nam);

	*s = ts;
	
	return error;
}

void *tcpaccept(void *socket, int *err)
{
    struct socket *s;
    int e = 0;
    struct sockaddr *nam;

    if( (e = tcp_accept_low((struct socket *)socket, &s, &nam)) == 0 ) {
        return s;
    }

    *err = e;

    return NULL;
}

void tcpdisconnect(void *socket)
{
    struct socket *so = (struct socket *)socket;

    sodisconnect(so);
}

int tcpisconnecting(void *socket)
{
    struct socket *so = (struct socket *)socket;

    return (so->so_state & SS_ISCONNECTING) && so->so_error == 0;
}

int tcpisconnected(void *socket)
{
    struct socket *so = (struct socket *)socket;

    return (so->so_state & SS_ISCONNECTED) > 0;
}

int nsc_setsockopt(void *socket, char *optname, void *val, size_t valsize)
{
    int opt;
    struct socket *so = (struct socket *)socket; 
    struct sockopt sopt;

    if( strcmp(optname, "SO_SNDBUF") == 0 ) opt = SO_SNDBUF;
    else if( strcmp(optname, "SO_RCVBUF") == 0 ) opt = SO_RCVBUF;
    else return 1;
    
    sopt.sopt_name = opt;
    sopt.sopt_dir = SOPT_SET;
    sopt.sopt_level = SOL_SOCKET;
    sopt.sopt_val = val;
    sopt.sopt_valsize = valsize;
    sopt.sopt_td = NULL;
    
    return sosetopt(so, &sopt);
}

void tcpprintstate(void *socket)
{
    struct socket *so = (struct socket *)socket;
    struct inpcb *inp;
    struct tcpcb *tp;


    inp = sotoinpcb(so);
    if (inp == 0) {
        return;
    }
    tp = intotcpcb(inp);

#if 0
    struct tcpcb {
        struct  tsegqe_head t_segq;
        int     t_dupacks;              /* consecutive dup acks recd */
        struct  tcptemp *unused;        /* unused */

        struct  callout *tt_rexmt;      /* retransmit timer */
        struct  callout *tt_persist;    /* retransmit persistence */
        struct  callout *tt_keep;       /* keepalive */
        struct  callout *tt_2msl;       /* 2*msl TIME_WAIT timer */
        struct  callout *tt_delack;     /* delayed ACK timer */

        struct  inpcb *t_inpcb;         /* back pointer to internet pcb */
        int     t_state;                /* state of this connection */
        u_int   t_flags;
#define TF_ACKNOW       0x00001         /* ack peer immediately */
#define TF_DELACK       0x00002         /* ack, but try to delay it */
#define TF_NODELAY      0x00004         /* don't delay packets to coalesce */
#define TF_NOOPT        0x00008         /* don't use tcp options */
#define TF_SENTFIN      0x00010         /* have sent FIN */
#define TF_REQ_SCALE    0x00020         /* have/will request window scaling */
#define TF_RCVD_SCALE   0x00040         /* other side has requested scaling */
#define TF_REQ_TSTMP    0x00080         /* have/will request timestamps */
#define TF_RCVD_TSTMP   0x00100         /* a timestamp was received in SYN */
#define TF_SACK_PERMIT  0x00200         /* other side said I could SACK */
#define TF_NEEDSYN      0x00400         /* send SYN (implicit state) */
#define TF_NEEDFIN      0x00800         /* send FIN (implicit state) */
#define TF_NOPUSH       0x01000         /* don't push */
#define TF_REQ_CC       0x02000         /* have/will request CC */
#define TF_RCVD_CC      0x04000         /* a CC was received in SYN */
#define TF_SENDCCNEW    0x08000         /* send CCnew instead of CC in SYN */
#define TF_MORETOCOME   0x10000         /* More data to be appended to sock */
#define TF_LQ_OVERFLOW  0x20000         /* listen queue overflow */
#define TF_LASTIDLE     0x40000         /* connection was previously idle */
#define TF_RXWIN0SENT   0x80000         /* sent a receiver win 0 in response */
        int     t_force;                /* 1 if forcing out a byte */

        tcp_seq snd_una;                /* send unacknowledged */
        tcp_seq snd_max;                /* highest sequence number sent;
                                         * used to recognize retransmits
                                         */
        tcp_seq snd_nxt;                /* send next */
        tcp_seq snd_up;                 /* send urgent pointer */

        tcp_seq snd_wl1;                /* window update seg seq number */
        tcp_seq snd_wl2;                /* window update seg ack number */
        tcp_seq iss;                    /* initial send sequence number */
        tcp_seq irs;                    /* initial receive sequence number */

        tcp_seq rcv_nxt;                /* receive next */
        tcp_seq rcv_adv;                /* advertised window */
        u_long  rcv_wnd;                /* receive window */
        tcp_seq rcv_up;                 /* receive urgent pointer */

        u_long  snd_wnd;                /* send window */
        u_long  snd_cwnd;               /* congestion-controlled window */
        u_long  snd_bwnd;               /* bandwidth-controlled window */
        u_long  snd_ssthresh;           /* snd_cwnd size threshold for
                                         * for slow start exponential to
                                         * linear switch
                                         */
        u_long  snd_bandwidth;          /* calculated bandwidth or 0 */
        tcp_seq snd_recover;            /* for use in NewReno Fast Recovery */
        tcp_seq snd_high;               /* for use in NewReno Fast Recovery */

        u_int   t_maxopd;               /* mss plus options */

        u_long  t_rcvtime;              /* inactivity time */
        u_long  t_starttime;            /* time connection was established */
        int     t_rtttime;              /* round trip time */
        tcp_seq t_rtseq;                /* sequence number being timed */

        int     t_bw_rtttime;           /* used for bandwidth calculation */
        tcp_seq t_bw_rtseq;             /* used for bandwidth calculation */

        int     t_rxtcur;               /* current retransmit value (ticks) */
        u_int   t_maxseg;               /* maximum segment size */
        int     t_srtt;                 /* smoothed round-trip time */
        int     t_rttvar;               /* variance in round-trip time */

        int     t_rxtshift;             /* log(2) of rexmt exp. backoff */
        u_int   t_rttmin;               /* minimum rtt allowed */
        u_int   t_rttbest;              /* best rtt we've seen */
        u_long  t_rttupdated;           /* number of times rtt sampled */
        u_long  max_sndwnd;             /* largest window peer has offered */

        int     t_softerror;            /* possible error not yet reported */
        /* out-of-band data */
        char    t_oobflags;             /* have some */
        char    t_iobc;                 /* input character */
#define TCPOOB_HAVEDATA 0x01
#define TCPOOB_HADDATA  0x02
        /* RFC 1323 variables */
        u_char  snd_scale;              /* window scaling for send window */
        u_char  rcv_scale;              /* window scaling for recv window */
        u_char  request_r_scale;        /* pending window scaling */
        u_char  requested_s_scale;
        u_long  ts_recent;              /* timestamp echo data */

        u_long  ts_recent_age;          /* when last updated */
        tcp_seq last_ack_sent;
        /* RFC 1644 variables */
        tcp_cc  cc_send;                /* send connection count */
        tcp_cc  cc_recv;                /* receive connection count */
        /* experimental */
        u_long  snd_cwnd_prev;          /* cwnd prior to retransmit */
        u_long  snd_ssthresh_prev;      /* ssthresh prior to retransmit */
        tcp_seq snd_high_prev;          /* snd_high prior to retransmit */
        u_long  t_badrxtwin;            /* window for retransmit recovery */
        u_char  snd_limited;            /* segments limited transmitted */
    };
#endif

    /*fprintf(out, 
      "snd_una: %u snd_max: %u snd_nxt: %u snd_up: %u snd_wl1: %u "
      "snd_wl2: %u iss: %u irs:%u\n"
      "rcv_nxt: %u rcv_wnd: %u rcv_up: %u rcv_adv: %u\n"		
      "snd_wnd: %u snd_cwnd: %u snd_bwnd: %u snd_ssthresh: %u\n"
      "snd_bandwidth: %u snd_recover: %u snd_high: %u\n"
      "t_maxopd: %u t_rcvtime: %u t_starttime: %u t_rtttime: %u\n"
      ,
      tp->snd_una, tp->snd_max, tp->snd_nxt, tp->snd_up, tp->snd_wl1,
      tp->snd_wl2, tp->iss, tp->irs,
      tp->rcv_nxt, tp->rcv_wnd, tp->rcv_up, tp->rcv_adv,
      tp->snd_wnd, tp->snd_cwnd, tp->snd_bwnd, tp->snd_ssthresh,
      tp->snd_bandwidth, tp->snd_recover, tp->snd_high,
      tp->t_maxopd, tp->t_rcvtime, tp->t_starttime, tp->t_rtttime
      );*/
    /* tcp_seq rcv_nxt;                
       tcp_seq rcv_adv;          
       u_long  rcv_wnd;         
       tcp_seq rcv_up;         

       u_long  snd_wnd;           
       u_long  snd_cwnd;         
       u_long  snd_bwnd;        
       u_long  snd_ssthresh;   

*/
    printf(
            "%d: rtt: %d srtt: %d rttvar: %d rxtcur: %d rttmin: %u "
            "rxtshift: %d rttbest: %u rttupdated: %lu t_rtttime: %d"
            "\n", 
            get_stack_id(), 
            tp->t_rtttime, tp->t_srtt, tp->t_rttvar, tp->t_rxtcur, 
            tp->t_rttmin,
            tp->t_rxtshift, tp->t_rttbest, tp->t_rttupdated, tp->t_rtttime
          );
    
}

extern int hz;

int nsc_get_tcp_var(void *socket, const char *var, char *result, int rlen)
{
    // Perform common functions: get at a TCP structure:
    struct socket *so = (struct socket *)socket;
    struct inpcb *inp;
    struct tcpcb *tp;
    float t_to_s = 1.0f / (float)hz; // ticks to seconds

    inp = sotoinpcb(so);
    if (inp == 0) {
        return 0;
    }
    tp = intotcpcb(inp);
    // --- 

         if(strcmp(var, "srtt_") == 0)  // in ticks
    {
        snprintf(result, rlen, "%f", (float)tp->t_srtt / 
                (float)TCP_RTT_SCALE);
        return 1;
    }
    else if(strcmp(var, "rttvar_") == 0) // in ticks
    {
        snprintf(result, rlen, "%f", (float)tp->t_rttvar / 
                (float)TCP_RTTVAR_SCALE);
        return 1;
    }
    /*
    // Don't really care about backoff for now; if we ever do the code below
    // almost works. Probably just need to find out where tcp_backoff is
    // defined.
    else if(strcmp(var, "backoff_") == 0) 
    {
        snprintf(result, rlen, "%lu", tcp_backoff[tp->t_rxtshift]);
        return 1;
    }*/
    else if(strcmp(var, "cwnd_") == 0)
    {
        snprintf(result, rlen, "%lu", tp->snd_cwnd);
        return 1;
    }
    else if(strcmp(var, "ssthresh_") == 0)
    {
        snprintf(result, rlen, "%lu", tp->snd_ssthresh);
        return 1;
    }
    else if(strcmp(var, "seqno_") == 0)
    {
        snprintf(result, rlen, "%lu", (u_long)(tp->snd_nxt - tp->iss));
        return 1;
    }
    else if(strcmp(var, "ack_") == 0)
    {
        snprintf(result, rlen, "%lu", (u_long)(tp->rcv_nxt - tp->irs));
        return 1;
    }
    else if(strcmp(var, "rxtcur_") == 0)
    {
        snprintf(result, rlen, "%f", (float)tp->t_rxtcur);
        return 1;
    }

    return 0;
}

void timer_expire(void *cc)
{
    struct callout *c = (struct callout *)cc;

    c->c_flags &= ~CALLOUT_PENDING;

    if(c->c_func) {
        c->c_func(c->c_arg);
    }
}
