/*
    globaliser -- programmatically replace globals in C source code
    Copyright (C) 2003-2005 Sam Jansen

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the Free
    Software Foundation; either version 2 of the License, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc., 59
    Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
// $Id: handle_global.cc 1662 2008-01-10 19:02:40Z stj2 $ 
#include <string>
#include <list>
#include <set>
#include <map>
#include <iostream>
#include <algorithm>
#include <iterator>
#include <stdio.h>
#include <assert.h>

#include "../sim/num_stacks.h"

using namespace std;
extern int opt_verbose;

struct SGlobal
{
    string s_identifier;
    string s_function;
    string s_mangled;
    enum Command {
        t_normal,
        t_section,
        t_section_stopstart,
    } e_cmd;
    bool b_expanded_decl;
    

    SGlobal(const string &s_iden)
        : s_identifier(s_iden), s_mangled(s_iden), e_cmd(t_normal), 
          b_expanded_decl(false)
    {
    }

    SGlobal(const string &s_iden, Command cmd)
        : s_identifier(s_iden), s_mangled(s_iden), e_cmd(cmd),
          b_expanded_decl(false)
    {
    }
        

    SGlobal(const string &s_iden, const string &s_func)
        : s_identifier(s_iden), s_function(s_func), e_cmd(t_normal),
          b_expanded_decl(false)
    {
        s_mangled = s_func + string("/") + s_iden;
    }

    bool operator ==(const SGlobal &g) const {
        if(s_mangled.compare(g.s_mangled) == 0)
            return true;
        return false;
    }
};

struct Symbol
{
    string s_mangled_iden;
    bool b_extern;
    bool b_volatile;
    bool b_unbounded_array;

    Symbol(const string &iden, bool ext, bool vol, bool unb_arr)
        : s_mangled_iden(iden), b_extern(ext), b_volatile(vol), 
        b_unbounded_array(unb_arr)
    {
    }
};

struct SGlobalComparitor
{
    size_t operator ()(const SGlobal &a, const SGlobal &x) const {
        return strcmp(a.s_mangled.c_str(), x.s_mangled.c_str()) < 0;
    }
};


static string global_reference(const string &iden);
static string global_reference_toplevel(const string &iden, int);
string global_old_reference(const string &iden);
bool global_expand_decl(const string &iden, const string &funcname);

typedef map< string, list<Symbol> > SymbolMap;
typedef set< SGlobal, SGlobalComparitor > GlobalSet;

SymbolMap symbols;
GlobalSet globals;

void dump_symbols()
{
    SymbolMap::iterator i = symbols.begin();
    fprintf(stderr, "===== Symbol table:\n");
    for(; i != symbols.end(); ++i) {
        list<Symbol>::iterator j = i->second.begin();
        fprintf(stderr, "Symbol: '%s'\n", i->first.c_str());
        for(; j != i->second.end(); ++j) {
            fprintf(stderr, "\t%s\n", j->s_mangled_iden.c_str());
        }
    }
    fprintf(stderr, "===== End symbol table\n");
}

void dump_globals()
{
    GlobalSet::iterator i = globals.begin();

    fprintf(stderr, "----- Globals\n%20s : %20s : %20s\n",
            "Identifier", "Function", "Mangled");
    for(; i != globals.end(); ++i) {
        fprintf(stderr, "%20s : %20s : %20s\n",
                i->s_identifier.c_str(), i->s_function.c_str(), 
                i->s_mangled.c_str());
    }
    fprintf(stderr, "---- End globals\n");
}

void dump_string_list(const list< string > &l)
{
    copy(l.begin(), l.end(), ostream_iterator<string>(cout, "\n"));
}

void setup_globals(const char *filename)
{
    FILE *f;
    char buf[256];

    f = fopen(filename, "r");

    if(!f) {
        fprintf(stderr, "Unable to open globals file '%s'. Exiting.\n",
                filename);
        exit(2);
    }

    while(fgets(buf, 254, f)) {
        string t(buf);
        const char *ws = " \t\r\n";

        string::size_type start = t.find_first_not_of(ws);
        string::size_type end   = t.find_last_not_of(ws);

        if(start != string::npos && end != string::npos) {
            string mangled = t.substr(start, end - start + 1);

            ;
            
            if((start = mangled.find('/')) != string::npos) {
                // A static local variable (function/variable)
                string func = mangled.substr(0, start);
                string iden = mangled.substr(start + 1, string::npos);
                globals.insert(SGlobal(iden, func));
            } else if((start = mangled.find('$')) != string::npos) {
                // A special command, such as $section
                string iden = mangled.substr(0, start);
                string cmd = mangled.substr(start + 1, string::npos);

                if(cmd == "section") {
                    globals.insert(SGlobal(iden, SGlobal::t_section));
                } else if(cmd == "section_stopstart") {
                    string start = string("__start_set_") + iden,
                           stop  = string("__stop_set_") + iden;
                    
                    globals.insert(SGlobal(start, SGlobal::t_section_stopstart));
                    globals.insert(SGlobal(stop, SGlobal::t_section_stopstart));
                } else if (cmd == "expand_decl") {
                  SGlobal g(iden);
                  g.b_expanded_decl = true;
                  globals.insert(g);
                } else {
                    fprintf(stderr, "Warning: Unknown command '$%s', ignoring "
                            "global\n", cmd.c_str());
                }
            } else {
                globals.insert(SGlobal(mangled));
            }
        }
        
    }

    fclose(f);
}

bool is_global(const string& iden)
{
    return globals.find(SGlobal(iden)) != globals.end();
}

bool is_global_section(const string &iden)
{
    GlobalSet::const_iterator i = globals.find(SGlobal(iden));
    if(i != globals.end())
        return i->e_cmd == SGlobal::t_section;
    return false;
}

bool is_global_section_startstop(const string &iden)
{
    GlobalSet::const_iterator i = globals.find(SGlobal(iden));
    if(i != globals.end())
        return i->e_cmd == SGlobal::t_section_stopstart;
    return false;
}

bool is_static_local(const string &func, const string& iden)
{
    return globals.find(SGlobal(iden, func)) != globals.end();
}

bool report_global(const string &iden, bool ext)
{
    if(opt_verbose == 0)
        return false;

    bool ret = is_global(iden);
    if(ext && opt_verbose < 2)
        return ret;
    
    fprintf(stderr, "is_global? %s %s\n", iden.c_str(),
            ret ? "yes" : "no");
    return ret;
}

bool report_static_local(const string &func, const string& iden)
{
    if(opt_verbose == 0)
        return false;

    bool ret = is_static_local(func, iden);
    fprintf(stderr, "is_static_local? %s/%s %s\n", func.c_str(), 
            iden.c_str(), ret ? "yes" : "no");
    return ret;
}

extern "C" {
    extern int lineno;
    extern int charno;
    extern const char *progname;
}

void handle_global_reference(const string &funcname, string &buf, string &iden, 
        bool toplevel = false, int snum = 0)
{
    set<SGlobal>::const_iterator iter = globals.find(SGlobal(iden)),
        siter = globals.find(SGlobal(iden, funcname));

    if(iter != globals.end() || siter != globals.end()) {
        int s = buf.find(iden);
        buf.erase(s, iden.size());
        
        if(!toplevel) {
            SymbolMap::iterator i = symbols.find(iden.c_str());
            if(i == symbols.end()) {
                fprintf(stderr, "%s Error: line %d char: %d Cannot find "
                        "symbol '%s' (in function %s)\n"
                        "You probably have not declared the variable.\n"
                        , 
                        progname, lineno, charno,
                        iden.c_str(), funcname.c_str());
#if 1
                dump_symbols();
                dump_globals();
#endif
                exit(1);
            }
            bool new_method = i->second.front().b_unbounded_array ||
              global_expand_decl(iden, funcname);
            if(new_method) {
                buf.insert(s, global_reference(
                            i->second.front().s_mangled_iden));
            } else {
                buf.insert(s, global_old_reference(
                            i->second.front().s_mangled_iden)); 
            }
        } else {
            SymbolMap::iterator i = symbols.find(iden.c_str());

            if(i != symbols.end() && i->second.front().b_unbounded_array)
                buf.insert(s, global_reference_toplevel(iden, snum));
            else if(i != symbols.end())
                buf.insert(s, global_old_reference(
                            i->second.front().s_mangled_iden));
            else
                buf.insert(s, global_reference_toplevel(iden, snum));
        }
    }
}

bool global_expand_decl(const string &iden, const string &funcname)
{
  GlobalSet::const_iterator i = globals.find(SGlobal(iden));

  if (i == globals.end())
    i = globals.find(SGlobal(iden, funcname));

  return i->b_expanded_decl;
}

void global_define(const string &iden, const string &mangled, bool ext, bool vol, 
        bool unb_arr)
{
    SymbolMap::iterator i = symbols.find(iden);

    if(i != symbols.end()) {
        i->second.push_front(Symbol(mangled, ext, vol, unb_arr));
    } else {
        list<Symbol> l;
        l.push_back(Symbol(mangled, ext, vol, unb_arr));
        symbols.insert(SymbolMap::value_type(iden, l));
    }
}

void global_undefine(const string &iden)
{
    SymbolMap::iterator i = symbols.find(iden);

    if(i != symbols.end()) {
        i->second.pop_front();
    } else {
        assert(0);
    }
    
}

string global_get_best_iden(string &iden)
{
    SymbolMap::iterator i = symbols.find(iden);

    assert(i != symbols.end());

    // Return the last one defined (the first in the list):
    return i->second.front().s_mangled_iden;
}

string global_name_prefix(int stack_no)
{
    char buf[1024];
    string ret;
    snprintf(buf, 1024, "_GLOBAL_%d_", stack_no);
    ret.append(buf);
    return ret;
}

string global_name(const string &iden, int stack_no)
{
    char buf[1024];
    string ret;

    snprintf(buf, 1024, "%s%s_I", global_name_prefix(stack_no).c_str(), 
            iden.c_str());
    ret.append(buf);
    
    return ret;
}

// We assume 'iden' is the mangled name!
static string global_reference(const string &iden)
{
    char buf[1024];
    string ret;

    snprintf(buf, 1024, "(*%s[get_stack_id()])", iden.c_str());
    ret.append(buf);

    return ret;
}

static string global_reference_toplevel(const string &iden, int stack_no)
{
    return global_name(iden, stack_no);
}

string global_new_array_name(const string &iden)
{
    static int global_array_number = 0;
    char buf[1024];
    string ret;

    snprintf(buf, 1024, "_GLOBAL_%s_%d_A", iden.c_str(), global_array_number++);
    ret.append(buf);
    
    return ret;
}

string global_new_type_name()
{
    static int global_type_number = 0;
    char buf[1024];
    string ret;

    snprintf(buf, 1024, "_GLOBAL_%d_T", global_type_number);
    global_type_number++;
    ret.append(buf);

    return ret;
}

string global_old_name(const string &iden)
{
    return string("global_") + iden;
}

string global_old_reference(const string &iden)
{
    return iden + string("[get_stack_id()]");
}

string global_old_name_postfix(void)
{
    return string("[get_stack_id()]");
}

string global_old_name_postfix(int stack)
{
    char buf[1024];
    string ret;
    snprintf(buf, 1024, "[%d]", stack);
    ret.append(buf);
    return ret;
}

void global_replace_toplevel(string &init, int stack)
{
    string::size_type loc;
    string 
           glb = global_name_prefix(0),
           rbuf = global_name_prefix(stack);

    if(stack == 0)
        return;

    while((loc = init.find(glb)) != string::npos) {
        init.replace(loc, glb.size(), rbuf);
    }
}

void global_old_replace_toplevel(string &init, int stack)
{
    string::size_type loc;
    string glb = global_old_name_postfix(),
           rbuf = global_old_name_postfix(stack);

    // I think this is very unlikely, but might as well
    // leave the code here...
    while((loc = init.find(glb)) != string::npos) {
        init.replace(loc, glb.size(), rbuf);
    }
}
