/*
    globaliser -- programmatically replace globals in C source code
    Copyright (C) 2003-2006 Sam Jansen

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the Free
    Software Foundation; either version 2 of the License, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc., 59
    Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
struct node_t
{
    node_t()
    {
        init();

        s_token = token_text;
        s_buf = lex_text;
        s_ws = ws_text;
        s_attr = attr_text;
    }
    virtual ~node_t() {
        delete_children();
    }
    void delete_children() {
        list<node_t *>::iterator i = children.begin();
        for(; i != children.end(); ++i)
            delete *i;
        children.erase(children.begin(), children.end());
    }
    
    list<node_t *> children;
    string s_identifier;
    string s_buf;
    string s_struct;
    string s_ws;
    string s_attr;
    bool b_ignore;
    bool b_declaration_list;
    bool b_function_declaration;
    bool b_init_declarator;
    bool b_toplevel;
    
    bool b_typedef;
    bool b_static;
    bool b_extern;
    bool b_volatile;
    bool b_register;
    bool b_unbounded_array;
    
    int i_function_pointer;

    string s_token;


    /* New stuff for more intelligent declaration handling. */
    enum {
        t_general,
        t_declaration,
        t_init_declarator_list,
        t_init_declarator,
        t_declarator,
        t_direct_declarator,
        t_direct_declarator_array,
        t_function_declaration,
        t_parameter_type_list,
        t_typeof,
        t_statement_list,
        t_compound_statement,
        t_string,
        t_string_list,
    } type;


    bool check_compount_statement() const {
        if(type == t_compound_statement) return true;    
    
        list<node_t *>::const_iterator ci = children.begin();
        for(; ci != children.end(); ++ci)
            if((*ci)->check_compount_statement())
                return true;

        return false;
    }

    bool check_typedef() const {
        if(b_typedef) return true;    
    
        list<node_t *>::const_iterator ci = children.begin();
        for(; ci != children.end(); ++ci)
            if((*ci)->check_typedef())
                return true;

        return false;
    }

    bool check_ignore() const {
        if(b_ignore) return true;    
    
        list<node_t *>::const_iterator ci = children.begin();
        for(; ci != children.end(); ++ci)
            if((*ci)->check_ignore())
                return true;

        return false;
    }

    bool get_extern() const {
        if(b_extern) 
            return true;
        list<node_t *>::const_iterator ci = children.begin();
        for(; ci != children.end(); ci++)
            if((*ci)->get_extern())
                return true;
        return false;
    }

    bool get_volatile() const {
        if(b_volatile) 
            return true;
        list<node_t *>::const_iterator ci = children.begin();
        for(; ci != children.end(); ci++)
            if((*ci)->get_volatile())
                return true;
        return false;
    }

    bool get_static() const {
        if(b_static) 
            return true;
        list<node_t *>::const_iterator ci = children.begin();
        for(; ci != children.end(); ci++)
            if((*ci)->get_static())
                return true;
        return false;
    }

    bool get_register() const {
        if(b_register)
            return true;
        list<node_t *>::const_iterator ci = children.begin();
        for(; ci != children.end(); ci++)
            if((*ci)->get_register())
                return true;
        return false;
    }

    bool get_function_declaration_b() const {
        if(b_function_declaration) 
            return true;
        list<node_t *>::const_iterator ci = children.begin();
        for(; ci != children.end(); ci++)
            if((*ci)->get_function_declaration_b())
                return true;
        return false;
   }

    bool get_unbounded_array() const {
        if(b_unbounded_array) 
            return true;
        list<node_t *>::const_iterator ci = children.begin();
        for(; ci != children.end(); ci++)
            if((*ci)->get_unbounded_array())
                return true;
        return false;
    }

    bool identifiers_need_globalising(const string &func) const {
      if (!s_identifier.empty())
        if (is_global(s_identifier) || is_static_local(func, s_identifier))
          return true;
      
      list<node_t *>::const_iterator ci = children.begin();
      for(; ci != children.end(); ci++) {
        if ((*ci)->identifiers_need_globalising(func))
          return true;
      }
      return false;
    }

    node_t *get_identifier() {
        if(s_identifier.size() > 0)
            return this;
        
        list<node_t *>::iterator ci = children.begin();
        for(; ci != children.end(); ci++) {
            node_t *i = (*ci)->get_identifier();
            if(i) return i;
        }

        return NULL;
    }
    
    node_t *get_function_pointer() {
        if(i_function_pointer == 3)
            return this;

        list<node_t *>::iterator ci = children.begin();
        for(; ci != children.end(); ci++) {
            node_t *i = (*ci)->get_function_pointer();
            if(i) return i;
        }

        return NULL;
    }

    node_t *get_typeof() {
        if(type == t_typeof)
            return this;

        list<node_t *>::iterator ci = children.begin();
        for(; ci != children.end(); ci++) {
            node_t *i = (*ci)->get_typeof();
            if(i) return i;
        }

        return NULL;
    }
    
    void print() const {
        printf("%s", s_buf.c_str());

        list<node_t *>::const_iterator ci = children.begin();
        for(; ci != children.end(); ci++)
            (*ci)->print();

    }

    void print_to_str(string &dest) const {
        list<node_t *>::const_iterator ci = children.begin();
        for(; ci != children.end(); ci++)
            (*ci)->print_to_str(dest);

        // TODO: use an ostringstream for efficiency
        dest += s_buf;
    }

    void init() {
        b_ignore = b_typedef = b_declaration_list = b_function_declaration 
            = b_init_declarator = b_toplevel = b_extern = b_volatile
            = b_static = b_register = b_unbounded_array = false;
        i_function_pointer = 0;

        type = t_general;
    }

    bool handle_global(string func = "", list<string> *defined_globals = NULL) {
        if(get_function_declaration_b())
            return false;
        if(!identifiers_need_globalising(func))
            return false;
        
        string result;
        list<node_t *>::iterator ci = children.begin();
        node_t *declaration_specifiers = *ci; ++ci;
        if(children.size() == 4) ++ci;
        node_t *init_declarator_list = *ci; ++ci;
        node_t *semicolon = *ci;
        string orig_node_output;
        int orig_newlines;
        
        assert(declaration_specifiers && init_declarator_list && semicolon);

        print_to_str(orig_node_output);
        orig_newlines = count(orig_node_output.begin(), 
                orig_node_output.end(), '\n');

        bool is_extern = declaration_specifiers->get_extern();
        bool is_volatile = declaration_specifiers->get_volatile();
        bool is_static = declaration_specifiers->get_static();
        bool is_register = declaration_specifiers->get_register();
        node_t *fp = get_function_pointer();
        string global_type_name = global_new_type_name();

        // XXX: Function pointers are globalised incorrectly. Most code uses
        // a typedef for function pointers, which means this doesn't really
        // come up in practice. However it should be fixed by either 
        // integrating the below code into what follows it, or porting the new
        // method of changing globals over for function pointers as well.
        if(fp) {
            // Function pointer. We need a special case for function pointers,
            // nothing much we can do about it. Stupid functions pointers,
            // they just don't fit in with the rest of the grammar well.
            node_t *iden = fp->get_identifier();
            node_t *id = init_declarator_list;
            string orig_iden = iden->s_identifier;
            int stack;

            if(is_static && func.size()) 
                report_static_local(func, orig_iden);
            else 
                report_global(orig_iden, is_extern);

            if( !( (!(is_static && func.size()) && is_global(orig_iden)) 
                    || (is_static && func.size() 
                            && is_static_local(func, orig_iden))) )
                return false;

            // We should be using the new method here...
            string::size_type loc;
            string new_iden = global_old_name(orig_iden);

            loc = iden->s_buf.find(orig_iden);
            assert(loc != string::npos);
            iden->s_buf.replace(loc, orig_iden.size(), new_iden 
                    + string("[NUM_STACKS]"));

            global_define(orig_iden, new_iden, is_extern, is_volatile, false);
            if(defined_globals) {
                
                defined_globals->push_back(orig_iden);
            }

            declaration_specifiers->print_to_str(result);

            // XXX:
            // Can you even have a list of initialised function pointers?
            // If so, I don't think anybody ever does. It's easier to typedef
            // the function pointer in such a case because the syntax is ugly.
            // So we don't even bother checking for a list of identifiers
            // here.
            // Answer: yes you can. This should be fixed.
            if(id->type == t_init_declarator_list) {
                assert(0);
            }

            if(id->type == t_init_declarator) {
                id = id->children.front();
            }

            id->print_to_str(result);

            // TODO: make the function pointer initialise correctly.
            if(id->type == t_init_declarator) {
                node_t *initialiser_node = id->children.back();
                string init_str;
                    
                initialiser_node->print_to_str(init_str);
                    
                result += " = {\n";
                for(stack = 0; stack < NUM_STACKS; stack++) {
                    string is = init_str;

                    global_replace_toplevel(is, stack);
                    global_old_replace_toplevel(is, stack);
                    
                    result += is;
                    result += ",\n";
                }
                result += " }";
            }

            result += ";\n";

            delete_children();
            s_buf = result;

            return true;
        }
        // END FUNCTION POINTER STUFF -------------------------------------

        string ws = get_whitespace(); remove_whitespace();
        string storage_class_specifiers;

        string attr = declaration_specifiers->find_attr_and_remove();

        if(is_extern) {
            declaration_specifiers->remove_node("extern");
            storage_class_specifiers += "extern ";
        }
        if(is_volatile) {
            declaration_specifiers->remove_node("volatile");
            storage_class_specifiers += "volatile ";
        }
        if(is_static) {
            declaration_specifiers->remove_node("static");
            storage_class_specifiers += "static ";
        }
        if(is_register) {
            declaration_specifiers->remove_node("register");
            storage_class_specifiers += "register ";
        }

        if (declaration_specifiers->remove_node("__inline__") ||
            declaration_specifiers->remove_node("__inline") ||
            declaration_specifiers->remove_node("inline__")) {
          storage_class_specifiers += "__inline__ ";
        }

        if (!attr.empty()) {
            storage_class_specifiers += " "; // TODO: we might be adding more whitespace than we need here, check this
            storage_class_specifiers += attr;
        }
        
        declaration_specifiers->reduce();
        result = "typedef " + declaration_specifiers->s_buf;
        result += string(" ") + global_type_name + string("; ");
        // typedef done.
       
        // Find a list of init_declarators:
        list<node_t *> init_declarators;
        node_t *idl = init_declarator_list;
        do {
            if(idl->type == node_t::t_init_declarator_list) {
                init_declarators.push_front(idl->children.back());
                idl = idl->children.front();
                continue;
            }
            
            // Either an init_declarator or a declarator
            init_declarators.push_front(idl);
            idl = NULL;
        } while(idl);
                
        // Produce the code to declare and possibly initialise everything in
        // the init_declarators list.
        // We need to check each identifier and see if it is a global variable
        // here.
        list<node_t *>::iterator id_iter = init_declarators.begin();
        for(; id_iter != init_declarators.end(); ++id_iter) {
            node_t *id = *id_iter, *i, *pointer = NULL, *array = NULL,
                *initialiser_node = NULL, *declarator = NULL;
            int stack;
            bool is_unbounded_array = false;

            // init_declarator:
            if(id->type == node_t::t_init_declarator 
                    && id->children.size() == 3) {
                // declarator equals_t initializer
                initialiser_node = id->children.back();
                i = id->children.front();
            } else {
                // declarator
                i = id;
            }
            declarator = i;
            // Iteratate past pointer nodes
            if(i->type == node_t::t_declarator) {
                pointer = i->children.front();
                i = i->children.back();
            }
            // Iterate past array nodes
            while(i->type == node_t::t_direct_declarator_array) {
                if(array == NULL)
                    array = i;
                i = i->children.front();
            }
            
            // It is possible to lose whitespace here; there could be
            // newlines between variables being declared. Lets hope that
            // is a non-issue.
            string orig_iden = i->get_identifier()->s_identifier;
            string type_name;

            type_name += storage_class_specifiers;
            type_name += " " + global_type_name + " ";

            if(array && array->get_unbounded_array()) {
                is_unbounded_array = true;
            }

            if(is_static && func.size())
                report_static_local(func, orig_iden);
            else
                report_global(orig_iden, is_extern);

            if( !( (!(is_static && func.size()) && is_global(orig_iden)) 
                    || (is_static && func.size() 
                            && is_static_local(func, orig_iden))) ) {

                if(initialiser_node) {
                    initialiser_node->hgr_initialiser(func.empty() ? 
                            "__toplevel__undefined" : func);
                    string init = initialiser_node->s_buf;

                    //global_replace_toplevel(init, 0);
                    global_old_replace_toplevel(init, 0);

                    initialiser_node->s_buf = init;
                }
                
                result += type_name;
                id->print_to_str(result);
                result += "; ";
                continue;
            }

            /* Special handling for section stuff; this should allow us to
             * globalise FreeBSD "linker sets" correctly. */
            if (is_global_section(orig_iden)) {
              node_t *attr_node = id->find_attribute_strings("__section__");
              if (!attr_node) {
                assert(storage_class_specifiers.find("__section__") != string::npos);
              }

              if (!attr_node) {
                assert(0); // TODO: handle this case by removing the section from the scs
                // and putting it into each var
                // At the moment we haven't implemented this case because it has
                // never come up
              }

              string orig_attr = attr_node->remove_attribute("__section__");
              string iden_attrs = i->find_attr_and_remove();

              if(initialiser_node)
                        initialiser_node->hgr_initialiser(func.empty() ? 
                            "__toplevel__undefined" : func);

              for(stack = 0; stack < NUM_STACKS; stack++) {
                result += type_name;
                if(pointer) pointer->print_to_str(result);
                result += string(" ");

                char stacknum[128];
                snprintf(stacknum, 128, "_global_section_%d_", stack);

                result += stacknum + orig_iden;

                string new_attr = orig_attr;
                
                {
                  string::size_type i, str_start, str_end;

                  i = new_attr.find("__section__");
                  i = new_attr.find("(", i+1); str_start = i + 1;
                  i = new_attr.find(")", i+1); str_end = i;

                  new_attr.insert(str_end, string(" \"") + stacknum + "\"");
                }

                result += " " + new_attr + " " + iden_attrs;

                result += " = ";

                string init_str;
                initialiser_node->print_to_str(init_str);

                global_replace_toplevel(init_str, stack);
                global_old_replace_toplevel(init_str, stack);

                result += init_str;
                result += ";";
              }
                    
              continue;
            } else {
              extern int opt_verbose; 
              if (opt_verbose > 0) {
                node_t *attr_node = id->find_attribute_strings("__section__");
                if (attr_node) {
                  fprintf(stderr, "Warning: global '%s' has __section__ "
                      "attribute but is not handled with $section. Attribute "
                      "was: '%s'\n",
                      orig_iden.c_str(), attr_node->s_attr.c_str());
                }
              }
            }

            if(is_global_section_startstop(orig_iden)) {
                // Forward declare all the actual symbols
                // (__start_set_[name]_global_section_[number]_)
                result += type_name;
                for(stack = 0; stack < NUM_STACKS; stack++) {
                    if(pointer) pointer->print_to_str(result);
                    
                    char gname[128];
                    snprintf(gname, 128, "_global_section_%d_", stack);

                    result += orig_iden + gname;

                    if(stack != NUM_STACKS - 1)
                        result += ",";
                }
                result += ";  ";

                // Make sure we declare the array as static and not extern
                type_name.erase(type_name.find("extern"), 6);
                type_name = string("static ") + type_name;
                
                // And create the array, much like we do for "unbounded arrays"
                result += type_name;
                if(pointer) pointer->print_to_str(result);
                result += string(" * ") + orig_iden + string("[NUM_STACKS] = { ");

                for(stack = 0; stack < NUM_STACKS; stack++) {
                    // "__start_set_sysinit_set" "_global_section_0_"
                    char gname[128];
                    snprintf(gname, 128, "_global_section_%d_", stack);

                    result += string("&") + orig_iden + gname;

                    result += ", ";
                }
                result += " };    ";

                // Define the global variable
                global_define(orig_iden, orig_iden, false, false, true);
                
                continue;
            }

            string iden_attrs = i->find_attr_and_remove();

            bool use_old_method = !is_unbounded_array;

            if (global_expand_decl(orig_iden, func)) {
              use_old_method = false;
            }

            if(use_old_method) {
                // Use old method
                string new_iden = global_old_name(orig_iden);
                result += type_name;
                if(pointer) pointer->print_to_str(result);
                result += string(" ") + new_iden + string("[NUM_STACKS]");

                if (!iden_attrs.empty())
                  result += string(" ") + iden_attrs;
                
                // Need to do this before initialiser_node->hgr_initialiser()
                global_define(orig_iden, new_iden, is_extern, is_volatile, 
                        false);
                if(defined_globals) {
                    
                    defined_globals->push_back(orig_iden);
                }
                
                if(initialiser_node) {
                    initialiser_node->hgr_initialiser(func.empty() ? 
                            "__toplevel__undefined" : func);

                    string init_str;
                    initialiser_node->print_to_str(init_str);
                    
                    result += string(" = { ");
                    for(stack = 0; stack < NUM_STACKS; stack++) {
                        string is = init_str;

                        global_replace_toplevel(is, stack);
                        global_old_replace_toplevel(is, stack);
                        
                        result += is;
                        result += ", ";
                    }
                    result += string(" }");
                }
                result += "; ";
                continue;
            }

            string orig_init;
            if(initialiser_node) {
                initialiser_node->hgr_initialiser(func.empty() ? 
                            "__toplevel__undefined" : func);
                orig_init = initialiser_node->s_buf;
            }
            for(stack = 0; stack < NUM_STACKS; stack++) {
                result += type_name;
                i->s_buf = string(" ") + global_name(orig_iden, stack);

                if (!iden_attrs.empty())
                  result += string(" ") + iden_attrs;

                // modify initialiser_node so it refers to the correct global
                // instances.
                if(initialiser_node)
                {
                    string init = orig_init; 
                    global_replace_toplevel(init, stack);
                    global_old_replace_toplevel(init, stack);

                    initialiser_node->s_buf = init;
                }

                id->print_to_str(result); 
                result += "; ";
            }
            
            // make array
            result += make_global_array(orig_iden, is_extern, is_volatile);
        }

        delete_children();
        s_buf = result + ws;

        return true;
    }

private:
    string make_global_array(string iden, bool is_extern, bool is_volatile) {
        string globalname = global_name(iden, 0);
        string result;
        int stack;

        result += "static ";
        result += "__attribute__ ((unused)) ";
        result += string("__typeof__(") + globalname + ") ";

        globalname = global_new_array_name(iden);
        
        result += string("*") + globalname + "[NUM_STACKS]";

        global_define(iden, globalname, is_extern, is_volatile, true);
        
        result += " = { ";

        for(stack = 0; stack < NUM_STACKS; stack++) {
            result += "&";
            result += global_name(iden, stack);
            result += ", ";
        }
        result += "};";

        return result;
    }

    string get_whitespace() {
        // the token is always at the end of the whitespace.
        string ws = s_ws;

        list<node_t *>::iterator ci = children.begin();
        for(; ci != children.end(); ++ci) {
            ws += (*ci)->get_whitespace();
        }
        return ws;
    }

    void remove_whitespace() {
        list<node_t *>::iterator ci = children.begin();
        for(; ci != children.end(); ++ci) {
            (*ci)->remove_whitespace();
        }
        s_buf = string(" ") + s_token;
        if (!s_attr.empty()) {
          s_buf += string(" ") + s_attr;
        }
    }

    string find_attr_and_remove() {
        std::ostringstream oss;
        list<node_t *>::iterator ci = children.begin();
        for(; ci != children.end(); ++ci) {
          oss << (*ci)->find_attr_and_remove();
        }
        // We can't both have children with attrs, and have attrs ourself
        // (only leafs can have attributes)
        assert(oss.str().size() == 0 || s_attr.size() == 0);
        if (s_attr.size() > 0) {
          string::size_type s = s_buf.find(s_attr);
          assert (s != string::npos);
          s_buf.erase(s, s_attr.size());
          string ret = s_attr;
          s_attr.clear();
          return ret + " ";
        }
        return oss.str();
    }

    string remove_attribute(const string& attr) {
      if (s_attr.find(attr) != string::npos) {
        string::size_type a_start, aa_start, i;
        
        a_start = s_attr.find(attr);
        aa_start = s_attr.rfind("__attribute", a_start);

        int depth = 0;
        for (i = aa_start; i < s_attr.size(); i++) 
        {
          if (s_attr[i] == '(') depth++;
          else if (s_attr[i] == ')') depth--;
          if (i > a_start && depth == 0)
            break;
        }

        string str = s_attr.substr(aa_start, i - aa_start);

        string::size_type buf_i = s_buf.find(str);
        assert(buf_i != string::npos);

        s_buf.erase(buf_i, str.size());
        s_attr.erase(aa_start, i - aa_start);

        return str;
      }

      list<node_t *>::iterator ci = children.begin();
      for(; ci != children.end(); ++ci) {
        string s = (*ci)->remove_attribute(attr);
        if (!s.empty())
          return s;
      }
      return string();
    }

    bool has_attribute() {
      if (!s_attr.empty())
        return true;

      list<node_t *>::iterator ci = children.begin();
      for(; ci != children.end(); ++ci) {
        if ((*ci)->has_attribute())
          return true;
      }
      return false;
    }
    void print_attrs() {
      list<node_t *>::iterator ci = children.begin();
      for(; ci != children.end(); ++ci) {
        (*ci)->print_attrs();
      }
      fprintf(stderr, "'%s':'%s'\n", s_buf.c_str(), s_attr.c_str());
    }

    node_t *find_attribute_strings(const string& s) {
      if (!s_attr.empty()) {
        if (s_attr.find(s) != string::npos) {
          return this;
        }
      }
      list<node_t *>::iterator ci = children.begin();
      for(; ci != children.end(); ++ci) {
        node_t* attr = (*ci)->find_attribute_strings(s);
        if (attr)
          return attr;
      }
      return NULL;
    }
    
    bool remove_node(const string &token)
    {
        if(s_token.compare(token) == 0) {
            s_buf = s_token = "";
            return true;
        }

        list<node_t *>::iterator ci = children.begin();
        for(; ci != children.end(); ++ci) {
            if( (*ci)->remove_node(token) )
                return true;
        }
        return false;
    }

    void add_locals(string funcname, list<string> &locals, int &num_locals,
            list<string> &static_locals, int &num_static_locals) {

        if(type == t_declaration) {
            list<node_t *>::iterator ci = children.begin();
            node_t *declaration_specifiers = *ci; ++ci;
            node_t *init_declarator_list = *ci; ++ci;
            node_t *semicolon = *ci;
            
            if(!(children.size() == 3 && declaration_specifiers && init_declarator_list && semicolon))
            {
                /*fprintf(stderr, "Error! Danger! :: %s\n"
                        "%p %p %p\n"
                        "%s (%s) (%s)\n"
                        ,
                        funcname.c_str(),
                        declaration_specifiers, init_declarator_list,
                        semicolon,
                        s_token.c_str(),
                        s_identifier.c_str(),
                        s_buf.c_str()
                        
                        );*/
                // In this case the variable was an extern and handled earlier
                // XXX: we don't need to handle externs earlier anymore, they
                // can just be done here. 
                return;
            }


            bool is_static = declaration_specifiers->get_static();
            list<node_t *> init_declarators;
            node_t *idl = init_declarator_list;
            do {
                if(idl->type == node_t::t_init_declarator_list) {
                    init_declarators.push_front(idl->children.back());
                    idl = idl->children.front();
                    continue;
                }
                
                // Either an init_declarator or a declarator
                init_declarators.push_front(idl);
                idl = NULL;
            } while(idl);

            bool handle_static_locals = false;
            list<node_t *>::iterator ii = init_declarators.begin();
            for(; ii != init_declarators.end(); ++ii) {
                node_t *id = *ii, *i;

                if(id->type == node_t::t_init_declarator) {
                    i = id->children.front();
                } else {
                    i = id;
                }
                // Iteratate past pointer nodes
                if(i->type == node_t::t_declarator) {
                    i = i->children.back();
                }
                // Iterate past array nodes
                while(i->type == node_t::t_direct_declarator_array) {
                    i = i->children.front();
                }

                string &iden = i->get_identifier()->s_identifier;

                if(is_static && is_static_local(funcname, iden)) {
                    //static_locals.push_back(iden);
                    //num_static_locals++;
                    handle_static_locals = true;
                } else {
                    //fprintf(stderr, "adding local: '%s'\n", iden.c_str());
                    locals.push_back(iden); num_locals++;
                }
            }
            if(handle_static_locals) {
                //extern void dump_string_list(const list<string> &);

                int old_size = static_locals.size();
                handle_global(funcname, &static_locals);
                num_static_locals = static_locals.size() - old_size;
                
                //dump_string_list(static_locals);
            }
        } else if(type == t_parameter_type_list) {
            add_locals_simple(locals, num_locals);
        } else {
            list<node_t *>::iterator ci = children.begin();
            for(; ci != children.end(); ++ci)
                (*ci)->add_locals(funcname, locals, num_locals, static_locals,
                                  num_static_locals);
        }
    }

    void add_locals_simple(list<string> &locals, int &num_locals) {
        if(type == t_direct_declarator) {
            node_t *i = get_identifier();
            if(i) {
                locals.push_back(i->s_identifier);
                num_locals++;
            }
            return;
        }

        list<node_t *>::iterator ci = children.begin();
        for(; ci != children.end(); ++ci)
            (*ci)->add_locals_simple(locals, num_locals);
    }

    void hgr_initialiser(string func) {
        string orig_init;
        list<string> locals, static_locals;
        hgr(func, locals, static_locals, true, 0);
        print_to_str(orig_init);

        // More efficient form of reduce(). Or it should be.
        delete_children();
        s_buf = orig_init;
    }

    void hgr_toplevel(int snum = 0) {
        if(b_init_declarator) {
            list<string> l, sl;
            children.back()->hgr("__toplevel__undefined", l, sl, true, snum);
            return;
        } 
        
        list<node_t *>::iterator ci;
        for(ci = children.begin(); ci != children.end(); ++ci) {
            (*ci)->hgr_toplevel();
        }
    
        //if(s_identifier.size() > 0)
        //    ::handle_global_reference(s_buf, s_identifier, true);
    }

    /** Recursively handles global references. This should be initially called
        on a function body, I believe. Then it will go over the code, noticing
        local variables that shadow global variables and changing the global
        variable references that need changing.
     */
    void hgr(string funcname, list<string> &locals, 
            list<string> &static_locals, bool toplevel = false, 
            int snum = 0) {
        if(b_ignore)
            return;

        if(b_toplevel) {
            hgr_toplevel();
            return;
        }

        //fprintf(stderr, "hgr:: '%s' locals.size()=%d\n", funcname.c_str(),
        //        locals.size());

        if(type == t_declaration) {
            if(!(children.size() > 1)) {
                //fprintf(stderr, "funcname: %s (%s)\n",
                //        funcname.c_str(), s_buf.c_str());
                return;
            }
            list<node_t *>::iterator i = children.begin();
            node_t *to = (*i)->get_typeof();
            if(to)
                to->hgr(funcname, locals, static_locals, toplevel, snum);

            i++;
            (*i)->hgr(funcname, locals, static_locals, toplevel, snum);
            return;
        }
            

        /* Old comment:
         *   if we are a compound statement, look for a declarator list
         *   first, then handle the statement list
         * New:
         *   Note that with the introduction of c99, variables can be declared
         *   anywhere in a compount statement. So we can no longer assume all
         *   declarations start in a "declarator_list" compound in the parse
         *   tree.
         */
        if(children.size() > 0) {
            if(type == t_statement_list) {
                node_t *statement = this;
                list<node_t *> right;
                while(statement->type == t_statement_list) {
                    right.push_back(statement->children.back());
                    statement = statement->children.front();
                }
                right.push_back(statement);

                //fprintf(stderr, "right.size()=%d\n", right.size());

                int num_locals = 0, num_static_locals = 0;
                while(!right.empty()) {
                    node_t *n = right.back(); right.pop_back();

                    if(n->check_compount_statement()) {
                        //fprintf(stderr, "handling compound statement...\n");
                        n->hgr(funcname, locals, static_locals, toplevel,
                                snum);
                    } else {
                        n->add_locals(funcname, locals, num_locals,
                                          static_locals, num_static_locals);
                        n->hgr(funcname, locals, static_locals, toplevel, 
                                   snum);
                    }
                }
                
                for(; num_locals > 0; num_locals--)
                    locals.pop_back();
                
                for(; num_static_locals > 0; num_static_locals--) {
                    assert(!static_locals.empty());
                    
                    global_undefine(static_locals.back());
                    static_locals.pop_back();
                }
            } else {
                // old code...    
                list<node_t *>::iterator ci;
                int num_locals = 0, num_static_locals = 0;
                for(ci = children.begin(); ci != children.end(); ++ci) {
                    if( (*ci)->b_declaration_list || (*ci)->type == t_declaration ) {
                        (*ci)->add_locals(funcname, locals, num_locals,
                                          static_locals, num_static_locals);
                        (*ci)->hgr(funcname, locals, static_locals, toplevel, 
                                   snum);
                    } else {
                        (*ci)->hgr(funcname, locals, static_locals, toplevel, 
                                   snum);
                    }
                }

                for(; num_locals > 0; num_locals--)
                    locals.pop_back();
                for(; num_static_locals > 0; num_static_locals--) {
                    global_undefine(static_locals.back());
                    static_locals.pop_back();
                }
            } 
        } else {
            
            if(s_identifier.size() > 0) {
                // variable reference!
                //fprintf(stderr, "Var ref: '%s'\n", s_identifier.c_str());
                list<string>::iterator si = locals.begin();
                for(; si != locals.end(); ++si) {
                    if( si->compare(s_identifier) == 0 ) {
                        return;
                    } 
                }
                //fprintf(stderr, "Handling ref.\n");
                ::handle_global_reference(funcname, s_buf, s_identifier, 
                        toplevel, snum);
            }
            
        }
    }

    void reduce() {
        list<node_t *>::iterator i = children.begin();
        string buf("");

        for(; i != children.end(); ++i) {
            (*i)->reduce();
            buf += (*i)->s_buf;
            delete *i;
        }
        children.erase(children.begin(), children.end());

        s_buf = buf + s_buf;
        s_token = s_buf;
    }

    node_t *get_function_declaration()
    {
        if(type == t_function_declaration)
            return this;
        
        list<node_t *>::iterator i = children.begin();
        for(; i != children.end(); ++i) {
            node_t *ret = (*i)->get_function_declaration();
            if(ret)
                return ret;
        }
        return NULL;
    }

public:
    void handle_global_references() {
        // Old notes I had before I'd written this code:
        // 
        // What we need to do:
        // Visit everything in the tree -- breadth first? depth first? hmm...
        // push locals onto a stack, visit children, pop locals from stack
        
        //      visiting: check for identifiers:
        //      if identifier: is it local? if so, continue
        //      else if it is global: handle_global_reference
        //      done
        //
        node_t *fn = get_function_declaration();
        if(!fn) {
            fprintf(stderr, "No fn::::\n");
            print();
            fprintf(stderr, ":=:=:=:\n\n");
        }

        list<string> locals, static_locals;
        hgr(fn->get_identifier()->s_identifier, locals, static_locals);
    }

    /* The ugly set of constructors that follow allow the code to be somewhat
     * nicer in parser.yy. Unfortunately C++ doesn't allow for a nice way of
     * variable parameters, the C-method isn't the nicest (not type-safe, need
     * some way of saying when the parameter list ends, etc.) */
    node_t(node_t *a) { 
        init();
        children.push_back(a);
    }
    node_t(node_t *a, node_t *b) { 
        init();
        children.push_back(a); children.push_back(b);
    }
    node_t(node_t *a, node_t *b, node_t *c) {
        init();
        children.push_back(a); children.push_back(b); children.push_back(c);
    }
    node_t(node_t *a, node_t *b, node_t *c, node_t *d) { 
        init();
        children.push_back(a); children.push_back(b); children.push_back(c); 
        children.push_back(d);
    }
    node_t(node_t *a, node_t *b, node_t *c, node_t *d, node_t *e) { 
        init();
        children.push_back(a); children.push_back(b); children.push_back(c); 
        children.push_back(d); children.push_back(e);
    }
    node_t(node_t *a, node_t *b, node_t *c, node_t *d, node_t *e, node_t *f) {
        init();
        children.push_back(a); children.push_back(b); children.push_back(c); 
        children.push_back(d); children.push_back(e); children.push_back(f);
    }
    node_t(node_t *a, node_t *b, node_t *c, node_t *d, node_t *e, node_t *f, 
            node_t *g) {
        init();
        children.push_back(a); children.push_back(b); children.push_back(c); 
        children.push_back(d); children.push_back(e); children.push_back(f); 
        children.push_back(g); 
    }
    node_t(node_t *a, node_t *b, node_t *c, node_t *d, node_t *e, node_t *f, 
            node_t *g, node_t *h) {
        init();
        children.push_back(a); children.push_back(b); children.push_back(c); 
        children.push_back(d); children.push_back(e); children.push_back(f); 
        children.push_back(g); children.push_back(h);
    }
    node_t(node_t *a, node_t *b, node_t *c, node_t *d, node_t *e, node_t *f, 
            node_t *g, node_t *h, node_t *i) {
        init();
        children.push_back(a); children.push_back(b); children.push_back(c); 
        children.push_back(d); children.push_back(e); children.push_back(f); 
        children.push_back(g); children.push_back(h); children.push_back(i);
    }
};


