/*
  Network Simulation Cradle
  Copyright (C) 2003-2005 Sam Jansen

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the Free
  Software Foundation; either version 2 of the License, or (at your option)
  any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc., 59
  Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

/* $Id$ */
#include <sys/param.h>
#include <sys/proc.h>
#include <sys/lock.h>
#include <sys/mutex.h>
#include <sys/kernel.h>
#include <sys/socket.h>
#include <sys/socketvar.h>

#include "support/support.h"
#include "num_stacks.h"

#include <sys/linker_set.h>
#include <sys/copyright.h>

#include "sim_errno.h"

struct sysinit **sysinit = NULL, **sysinit_end, **newsysinit, **newsysinit_end;
SET_DECLARE(sysinit_set, struct sysinit);

// exported from support/sort_sysinit.c:
extern void sort_sysinit(struct sysinit **start, struct sysinit **,struct sysinit **);
// ?

// Get rid of warnings by declaring some functions:
extern void init_systm();
extern void bzero(char*,int);
extern void printf(const char *,...);

//! Silly little copyright notice. In bright green!
char nsc_copyright[] =
"\033[1;32;40mNetwork Simulation Cradle\033[0m\n"
"Copyright (c) 2003 Sam Jansen.\n";

SYSINIT(placeholder, SI_SUB_DUMMY, SI_ORDER_ANY, NULL, NULL)
/** Used during kernel initialisation normally. We used it to print the
 * normal copyright notices + our own. */
static void
print_caddr_t(void *data __unused)
{
	printf("%s", (char *)data);
}
SYSINIT(announce, SI_SUB_COPYRIGHT, SI_ORDER_FIRST, print_caddr_t, copyright)
SYSINIT(version, SI_SUB_COPYRIGHT, SI_ORDER_SECOND, print_caddr_t, nsc_copyright)

void bzero(char *buf, int n)
{
    nsc_bzero(buf, n);
}

/** Call all the system intialisation functions (the sysinit set) */
void kern_init()
{
    register struct sysinit **sipp, **ne;
    // The following would normally happen in some vm setup stuff, we
    // dont include all the vm code (doesn't really make sense in
    // userspace).
    unsigned int size = kern_timeout_callwheel_alloc(NULL);
    kern_timeout_callwheel_alloc(malloc(size, 0, 0));
    kern_timeout_callwheel_init();

    // Lets initialise ticks to 1 -- 0 is often a special value and can be
    // interpreted in various undesirable ways. A machine would never have
    // ticks equal to 0 after booting up anyway.
    ticks = 1;

    // Taken from kern_main.c in FreeBSD 5.0
    if (sysinit == NULL) {
        sysinit = SET_BEGIN(sysinit_set);
        sysinit_end = SET_LIMIT(sysinit_set);
    }

    sort_sysinit(sysinit, sysinit_end, ne);
}

int cur_stack = 0;
int num_stacks = 0;

int new_stack_id()
{
	return num_stacks++;
}

int get_stack_id()
{
	return cur_stack;
}

void set_stack_id(int id)
{
	cur_stack = id;
}

/** Legacy function that was used when we needed locking for threads.
 * Has been left in for now, though it serves no purpose. */
void lock_stack()
{
}

/** Legacy function that has no purpose. */
void unlock_stack()
{
    assert(get_stack_id() != -1);
}

/* ---------------------------------------------------------------- */
/* Socket stuff
*/
#include <sys/uio.h>
#include <sys/socket.h>
#include <sys/socketvar.h>
#include <sys/protosw.h>
#include <netinet/in.h>


void nsc_socreate_udp(struct socket **so)
{
	assert(
	    socreate(AF_INET, so, SOCK_DGRAM, IPPROTO_UDP, proc0.p_ucred, 
            curthread)
        == 0);
}

void nsc_socreate_tcp(struct socket **so)
{
	assert(
	    socreate(AF_INET, so, SOCK_STREAM, IPPROTO_TCP, proc0.p_ucred, 
            curthread)
        == 0);
}

void nsc_soconnect(struct socket *so, unsigned int addr, int port)
{
	struct sockaddr_in s_dest;
	
	memset(&s_dest, 0, sizeof(struct sockaddr_in));
	s_dest.sin_family = AF_INET;	 	 
    s_dest.sin_len = sizeof(struct sockaddr_in);	 	 
    s_dest.sin_port = port;
    s_dest.sin_addr.s_addr = addr;
        
	soconnect(so, (struct sockaddr *)&s_dest, curthread);
}

void nsc_soclose(struct socket *so)
{
    soclose(so);
}

int nsc_sosend_blocking(struct socket *so, struct sockaddr_in *dest,
        const void *data, unsigned int datalen)
{

	struct msghdr msg;
	struct iovec aiov;
	struct uio auio;

	msg.msg_namelen = 0;
	msg.msg_name = (struct sockaddr *)dest;
	msg.msg_iov = &aiov;
	msg.msg_iovlen = 1;
	msg.msg_control = 0;
	msg.msg_flags = 0;
	
	aiov.iov_base = (void *)data;
	aiov.iov_len = datalen;

	auio.uio_iov = msg.msg_iov;
	auio.uio_iovcnt = msg.msg_iovlen;
	auio.uio_segflg = UIO_USERSPACE;
	auio.uio_rw = UIO_WRITE;
	auio.uio_td = curthread;
	auio.uio_offset = 0; 
	auio.uio_resid = aiov.iov_len;

	// Make sure the socket is blocking
	so->so_state &= ~SS_NBIO;

	return so->so_proto->pr_usrreqs->pru_sosend(
			so,					// struct socket *so
			msg.msg_name,		// struct sockaddr *addr
			&auio,				// struct uio *uio
			(struct mbuf *)0,	// struct mbuf *top
			(struct mbuf *)0,   // struct mbuf *control
			0,					// int flags 
			curthread);			// struct thread *td
}

int nsc_sosend(struct socket *so, struct sockaddr_in *dest, 
		const void *data, int *datalen)
{
	struct msghdr msg;
	struct iovec aiov;
	struct uio auio;
        int error;

	msg.msg_namelen = 0;
	msg.msg_name = (struct sockaddr *)dest;
	msg.msg_iov = &aiov;
	msg.msg_iovlen = 1;
	msg.msg_control = 0;
	msg.msg_flags = 0;
	
	aiov.iov_base = (void *)data;
	aiov.iov_len = *datalen;

	auio.uio_iov = msg.msg_iov;
	auio.uio_iovcnt = msg.msg_iovlen;
	auio.uio_segflg = UIO_USERSPACE;
	auio.uio_rw = UIO_WRITE;
	auio.uio_td = curthread;
	auio.uio_offset = 0;   
	auio.uio_resid = aiov.iov_len;

	// Make sure the socket is non-blocking, so we dont try and msleep()
	so->so_state |= SS_NBIO;

	error = so->so_proto->pr_usrreqs->pru_sosend(
			so,					// struct socket *so
			msg.msg_name,		// struct sockaddr *addr
			&auio,				// struct uio *uio
			(struct mbuf *)0,	// struct mbuf *top
			(struct mbuf *)0,   // struct mbuf *control
			0,					// int flags 
			curthread);			// struct thread *td
        
        *datalen = auio.uio_offset;

        return error;
}

int nsc_soreceive_blocking(struct socket *so, unsigned char *buf, int *buflen,
		struct sockaddr **from)
{

	struct msghdr msg;
	struct iovec aiov;
	struct uio auio;
	int error;
	
	msg.msg_namelen = 0;
	msg.msg_name = (caddr_t)0; // from ... ?
	msg.msg_iov = &aiov;
	msg.msg_iovlen = 1;
	msg.msg_control = 0;
	msg.msg_flags = 0;
	aiov.iov_base = buf;
	aiov.iov_len = *buflen;

	auio.uio_iov = msg.msg_iov;
	auio.uio_iovcnt = msg.msg_iovlen;
	auio.uio_segflg = UIO_USERSPACE;
	auio.uio_rw = UIO_READ; 
	auio.uio_td = curthread;
	auio.uio_offset = 0;      
	auio.uio_resid = *buflen;

	error = so->so_proto->pr_usrreqs->pru_soreceive(
				so, from, &auio, (struct mbuf **)0, 
				(struct mbuf **)0,
				&msg.msg_flags);

	buf = aiov.iov_base - auio.uio_offset;
	*buflen = auio.uio_offset;

	return error;
}

int nsc_soreceive(struct socket *so, unsigned char *buf, int *buflen,
		struct sockaddr **from)
{
	struct msghdr msg;
	struct iovec aiov;
	struct uio auio;
	int error;
	
	msg.msg_namelen = 0;
	msg.msg_name = (caddr_t)0; // from ... ?
	msg.msg_iov = &aiov;
	msg.msg_iovlen = 1;
	msg.msg_control = 0;
	msg.msg_flags = MSG_DONTWAIT; // DONTWAIT: nonblocking

	aiov.iov_base = buf;
	aiov.iov_len = *buflen;

	auio.uio_iov = msg.msg_iov;
	auio.uio_iovcnt = msg.msg_iovlen;
	auio.uio_segflg = UIO_USERSPACE;
	auio.uio_rw = UIO_READ; 
	auio.uio_td = curthread;
	auio.uio_offset = 0;      
	auio.uio_resid = aiov.iov_len;

    debugf("%s: buf:%p len:%d\n", __FUNCTION__, buf, *buflen);

	error = so->so_proto->pr_usrreqs->pru_soreceive(
				so, from, &auio, (struct mbuf **)0, 
				(struct mbuf **)0,
				&msg.msg_flags);

	buf = aiov.iov_base - auio.uio_offset;
	*buflen = auio.uio_offset;

	return error;
}

int nsc_getsockpeername(struct socket *so, void *addrspace, size_t *addrspclen, int *port, int peername)
{
	struct sockaddr *sa;
	int ret = peername ? so->so_proto->pr_usrreqs->pru_peeraddr(so, &sa) :
		so->so_proto->pr_usrreqs->pru_sockaddr(so, &sa);
	if (ret == 0) {
		if (sa->sa_family == AF_INET) {
			struct sockaddr_in *sin = (struct sockaddr_in *) sa;
			memcpy(addrspace, &sin->sin_addr, *addrspclen < sizeof(sin->sin_addr) ? *addrspclen : sizeof(sin->sin_addr));
			*port = sin->sin_port;
			*addrspclen = sizeof(sin->sin_addr);
		} else {
			abort();
		}
		free(sa);
	}
	return ret;
}

#include <sys/sysctl.h>
int nsc_sysctl(const char *sysctl_name, void *oldval, size_t *oldlenp,
    void *newval, size_t newlen)
{
    int retval = 0;
    int error = kernel_sysctlbyname(curthread, sysctl_name, oldval, oldlenp,
            newval, newlen, &retval);
    return error;
}


/*
 * Test whether the specified credentials imply "super-user" privilege.
 * Return 0 or EPERM.  The flag argument is currently used only to
 * specify jail interaction.
 */
int
suser_cred(struct ucred *cred, int flag)
/* [<][>][^][v][top][bottom][index][help] */
{
    return 0;
}


int nsc_convert_syserr_to_nscerr(int syserr)
{
/*
 * we assume 0 is a success indicator that doesn't require any mapping.
 * NSC_E* constants are always < 0.
 */
	if (syserr == 0)
		return syserr;

	switch (syserr) {
	case ERESTART: // fallthrough
	case EAGAIN: return NSC_EAGAIN;
	case EADDRINUSE: return NSC_EADDRINUSE;
	case EADDRNOTAVAIL: return NSC_EADDRNOTAVAIL;
	case EALREADY: return NSC_EALREADY;
	case ECONNREFUSED: return NSC_ECONNREFUSED;
	case ECONNRESET: return NSC_ECONNRESET;
	case EHOSTDOWN: return NSC_EHOSTDOWN;
	case EHOSTUNREACH: return NSC_EHOSTUNREACH;
	case EINPROGRESS: return NSC_EINPROGRESS;
	case EISCONN: return NSC_EISCONN;
	case EMSGSIZE: return NSC_EMSGSIZE;
	case ENETUNREACH: return NSC_ENETUNREACH;
	case ENOTCONN: return NSC_ENOTCONN;
	case ESHUTDOWN: return NSC_ESHUTDOWN;
	case ETIMEDOUT: return NSC_ETIMEDOUT;
	/* Error conditions that we should never see */
	case EPROTONOSUPPORT:
	case EDESTADDRREQ:
	case EPROTOTYPE:
	case ENOBUFS:
	case EOVERFLOW: nsc_assert(0, "network stack returned fatal error");
	}

	nsc_printf("%s: Unhandled error number %d\n", __func__, syserr);
	return NSC_EUNKNOWN;
}
