/*
  Network Simulation Cradle

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the Free
  Software Foundation; either version 2 of the License, or (at your option)
  any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc., 59
  Temple Place, Suite 330, Boston, MA 02111-1307 USA

  provides a /sbin/sysctl-like interface.
*/

#include <linux/string.h>
#include <linux/kernel.h>
#include <linux/types.h>

#include "../nsc/nsc.h"


long strtol(const char *cp, char **endp, unsigned int base);

/*
 * integer converters:
 * convert int to character representation, and vice versa.
 */
static size_t convert_strtoint(const char *value, void *res, size_t len)
{
	char *end;
	unsigned int i = strtol(value, &end, 10);

	if (*end != '\0')
		return 0;

	if (len > sizeof(i))
		len = sizeof(i);
	memcpy(res, &i, len);
	return sizeof(i);
}

static size_t convert_inttostr(const void *integer, size_t sizeof_int, char *out, size_t out_len)
{
	if (sizeof_int == sizeof(int)) {
		const int *ptr = integer;
		return snprintf(out, out_len, "%d", *ptr);
	}
	return 0;
}


/*
 * string 'converters'. Used by tcp_congestion_control.
 * these don;t convert anything and just pass the string value
 * between the 'kernel' sysctl-interface and the nsc caller.
 */
static size_t convert_str_in(const char *value, void *res, size_t len)
{
	return strlcpy(res, value, len);
}

/* same for sysctl-to-nsc case */
static size_t convert_str_out(const void *str, size_t str_len, char *out, size_t out_len)
{
	return strlcpy(out, str, out_len);
}


#define DECLARE_SYSCTL_INTVEC(name)	{ name, convert_strtoint, convert_inttostr }
#define DECLARE_SYSCTL_STRING(name)	{ name, convert_str_in, convert_str_out }
static struct {
	/* pointer to full name of sysctl, as printed by "/sbin/sysctl -a" */
	const char *name;
	/*
	 * converts *value and stores result in *mem.
	 * the result size (in bytes) is returned.
	 * 0: conversion failed.
	 * retval > mem_size: truncated and/or incomplete conversion.
	 */
	size_t (*convert_w)(const char *value, void *mem, size_t mem_size);

	/*
	 * converts a *mem blob returned from the sysctl call to human-readable form.
	 * retval > outbuf_len: result was truncated.
	 */
	size_t (*convert_r)(const void *mem, size_t mem_size, char *outbuf, size_t outbuf_len);
} nsc_linux_sysctl_map[] = {
/* Note: The cradle doesn't know about all of these yet. List created using:
last=""
egrep '(\.procname|\.proc_handler)' net/ipv4/sysctl_net_ipv4.c | awk '{print $3}' | tr -d '&,"' |
while read tok ; do
	case $tok in
	proc_dointvec|proc_dointvec_jiffies|proc_dointvec_minmax|ipv4_doint_and_flush|ipv4_sysctl_forward)
		echo -e \\\tDECLARE_SYSCTL_INTVEC\(\"net.ipv4.$last\"\)\,
	;;
	proc_tcp_congestion_control)
		echo -e \\\tDECLARE_SYSCTL_STRING\(\"net.ipv4.$last\"\)\,
	esac
	last=$tok
done
*/
	DECLARE_SYSCTL_INTVEC("net.ipv4.tcp_timestamps"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.tcp_window_scaling"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.tcp_sack"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.tcp_retrans_collapse"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.ip_forward"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.ip_default_ttl"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.ip_no_pmtu_disc"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.ip_nonlocal_bind"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.tcp_syn_retries"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.tcp_synack_retries"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.tcp_max_orphans"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.tcp_max_tw_buckets"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.ipfrag_high_thresh"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.ipfrag_low_thresh"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.ip_dynaddr"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.ipfrag_time"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.tcp_keepalive_time"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.tcp_keepalive_probes"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.tcp_keepalive_intvl"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.tcp_retries1"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.tcp_retries2"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.tcp_fin_timeout"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.tcp_syncookies"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.tcp_tw_recycle"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.tcp_abort_on_overflow"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.tcp_stdurg"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.tcp_rfc1337"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.tcp_max_syn_backlog"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.ip_local_port_range"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.icmp_echo_ignore_all"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.icmp_echo_ignore_broadcasts"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.icmp_ignore_bogus_error_responses"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.icmp_errors_use_inbound_ifaddr"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.igmp_max_memberships"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.igmp_max_msf"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.inet_peer_threshold"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.inet_peer_minttl"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.inet_peer_maxttl"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.inet_peer_gc_mintime"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.inet_peer_gc_maxtime"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.tcp_orphan_retries"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.tcp_fack"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.tcp_reordering"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.tcp_ecn"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.tcp_dsack"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.tcp_mem"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.tcp_wmem"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.tcp_rmem"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.tcp_app_win"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.tcp_adv_win_scale"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.icmp_ratelimit"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.icmp_ratemask"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.tcp_tw_reuse"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.tcp_frto"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.tcp_low_latency"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.ipfrag_secret_interval"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.ipfrag_max_dist"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.tcp_no_metrics_save"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.tcp_moderate_rcvbuf"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.tcp_tso_win_divisor"),
	DECLARE_SYSCTL_STRING("net.ipv4.tcp_congestion_control"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.tcp_abc"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.tcp_mtu_probing"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.tcp_base_mss"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.tcp_workaround_signed_windows"),
	DECLARE_SYSCTL_INTVEC("net.ipv4.tcp_slow_start_after_idle")
};


int nsc_sysctl_util_get(const char *name, char *value, size_t len)
{
	size_t i;

	for (i=0; i < ARRAY_SIZE(nsc_linux_sysctl_map); i++) {
		size_t buflen;
		int ret;
		char buf[512];
		if (strcmp(nsc_linux_sysctl_map[i].name, name) != 0)
			continue;
		buflen = sizeof(buf);
		ret = nsc_do_sysctl(name, buf, &buflen, NULL, 0);
		if (ret == 0)
			return nsc_linux_sysctl_map[i].convert_r(buf, buflen, value, len);
		return -1;
	}
	return -1;
}


int nsc_sysctl_util_set(const char *name, const char *value)
{
	size_t i;

	for (i=0; i < ARRAY_SIZE(nsc_linux_sysctl_map); i++) {
		size_t ret;
		char buf[64];
		if (strcmp(nsc_linux_sysctl_map[i].name, name) != 0)
			continue;
		ret = nsc_linux_sysctl_map[i].convert_w(value, buf, sizeof(buf));
		if (ret == 0) /* conversion failed */
			break;
		if (ret <= sizeof(buf))
			return nsc_do_sysctl(name, NULL, NULL, buf, ret);
		/* else could malloc() the needed amount, but sizeof(buf) should be large enough */
		return -1;
	}
	return -1;
}

int nsc_sysctl_util_getnum(size_t idx, char *name, size_t len)
{
	if (idx >= ARRAY_SIZE(nsc_linux_sysctl_map))
		return -1;

	return strlcpy(name, nsc_linux_sysctl_map[idx].name, len);
}

